#pragma once

#include <QScopedPointer>

/*

Class that represents the outcome of a function call.

It can either be a value or an exception.

*/
template <class T> class Try {
	private:
		QScopedPointer<T> value; // A pointer is used so T does not have to be default constructable
		std::exception_ptr exception;
		
		// Use Try::success and Try::failure to construct objects
		Try () {}

	public:
		Try (const Try<T>& other) {
			if (other.succeeded()) {
				value.reset(new T(*other.value));
			}
			else {
				exception = other.exception;
			}
		}

        void operator= (const Try<T>& other) {
            value.reset(other.succeeded() ? new T(*other.value) : nullptr);
            exception = other.exception;
        }
	
		bool succeeded () const {
			return exception == nullptr;
		}
		
		bool failed () const {
			return exception != nullptr;
		}

		// Return the value or throw the exception
		T getValue () const {
			if (exception) std::rethrow_exception(exception);
			return *value;
		}
		
		// Get the exception, null pointer in case of success
		std::exception_ptr getException () const {
			return exception;
		}
		
		static Try<T> success (const T& value) {
			Try<T> t;
			t.value.reset(new T(value));
			return t;
		}
		
		static Try<T> failure (std::exception_ptr exception) {
			Q_ASSERT(exception != nullptr);
			Try<T> t;
			t.exception = exception;
			return t;
		}
};

template <> class Try<void> {
	private:
		std::exception_ptr exception;
		
		// Use Try::success and Try::failure to construct objects
		Try (std::exception_ptr exception) : exception(exception) {}

	public:
		Try (const Try<void>& other) {
			exception = other.exception;
		}
	
		bool succeeded () const {
			return exception == nullptr;
		}
		
		bool failed () const {
			return exception != nullptr;
		}

		void getValue () const {
			if (exception) std::rethrow_exception(exception);
		}
		
		std::exception_ptr getException () const {
			return exception;
		}
		
		static Try<void> success () {
			return Try<void>(nullptr);
		}

		static Try<void> failure (std::exception_ptr exception) {
			Q_ASSERT(exception != nullptr);
			return Try<void>(exception);
		}
};
